package AnyEvent::HTTP::MultiGet;

use Modern::Perl;
use Moo;
use MooX::Types::MooseLike::Base qw(:all);
use namespace::clean;
use Data::Dumper;
BEGIN { 
  no namespace::clean;
  with 'Log::LogMethods','Data::Result::Moo';
  extends 'HTTP::MultiGet';
}

=head1 NAME

AnyEvent::HTTP::MultiGet - AnyEvent Loop Control Freindly LWP Like agent

=head1 SYNOPSIS

  use AnyEvent::HTTP::MultiGet;
  use AnyEvent::Loop;
  use Test::More;
  use Data::Dumper;

  my $self=AnyEvent::HTTP::MultiGet->new();
  my $count=0;
  TEST_LOOP: {
    my $req=HTTP::Request->new(GET=>'http://google.com');
    my @todo=HTTP::Request->new(GET=>'http://yahoo.com');
    push @todo,HTTP::Request->new(GET=>'http://news.com');
    my $total=2 + scalar(@todo);

    my $req_b=HTTP::Request->new(GET=>'https://127.0.0.1:5888');
    my $code;
    $code=sub {
      my ($obj,$request,$result)=@_;
      diag sprintf 'HTTP Response code: %i',$result->code;
      ++$count;
      if(my $next=shift @todo) {
        $self->add_cb($req,$code);
        $self->run_next;
      }
      no warnings;
      last TEST_LOOP if $total==$count;
    };
    $self->add_cb($req,$code);
    $self->add_cb($req_b,$code);
    $self->run_next;
    AnyEvent::Loop::run;
  }

=head1 DESCRIPTION

This class provides an AnyEvent::Loop::run frienddly implementation of HTTP::MultiGet.

=head1 OO Arguments and accessors

Arguemnts and object accessors:

  logger:          DOES(Log::Log4perl::Logger)
  request_opts:    See AnyEvent::HTTP params for details
  timeout:         Global timeout for everything 
  max_que_count:   How many requests to run at once 
  max_retry:       How many times to retry if we get a connection/negotiation error 

For internal use only: 

  in_control_loop: true when in the control loop
  stack:           Data::Queue object 
  que_count:       Total Number of elements active in the que
  retry:           Anonymous hash used to map ids to retry counts
  cb_map:          Anonymous hash used to map ids to callbacks

=cut

has cb_map=>(
  is=>'ro',
  isa=>HashRef,
  default=>sub { {} },
  required=>1,
);

# This method runs after the new constructor
sub BUILD {
  my ($self)=@_;
}

# this method runs before the new constructor, and can be used to change the arguments passed to the module
around BUILDARGS => sub {
  my ($org,$class,@args)=@_;
  
  return $class->$org(@args);
};

=head1 OO Methods

=over 4

=item * my $id=$self->add_cb($request,$code)

Adds a request with a callback handler.

=cut

sub add_cb {
  my ($self,$request,$code)=@_;
  my ($id)=$self->add($request);
  $self->cb_map->{$id}=[$code,$request];
  return $id;
}

sub que_function {
    my ($self,$req,$id)=@_;
    my $code=$self->SUPER::que_function($req,$id);

    return sub {
      $code->(@_);
      $self->_common_handle_callback($id);
      $self->run_next;
      $self->log_always("our que count is: ".$self->que_count);
    };
}

sub _common_handle_callback {
  my ($self,$id)=@_;
  if(exists $self->cb_map->{$id}) {
    if(exists $self->results->{$id}) {
      my ($code,$req)=@{delete $self->cb_map->{$id}};
      my $result=delete $self->results->{$id};
      my $response;
      if($result) {
        $response=$result->get_data;
      } else {
        $response=$self->RESPONSE_CLASS->new('',{Status=>500,Reason=>"Request Timed out"})->to_http_message;
      }
      $code->($self,$req,$response);
    }
  } else {
  }
}

sub block_for_ids {
  my ($self,@ids)=@_;
  my $result=$self->SUPER::block_for_ids(@ids);

  if($result) {
    foreach my $id (@ids) {
      $self->results->{$id}=$result->get_data->{$id};
      $self->_common_handle_callback($id);
      delete $self->results->{$id};
    }
  } else {
    foreach my $id (@ids) {
      $self->results->{$id}=$self->new_false("$result");
      $self->_common_handle_callback($id);
      delete $self->results->{$id};
    }
  }

  return $result;
}

=back

=head1 AUTHOR

Michael Shipper <AKALINUX@CPAN.ORG>

=cut

1;
