#include "dtatwCommon.h"
#include "dtatwExpat.h"
#include "fcntl.h"

/*======================================================================
 * Globals
 */

cxStoredRecord cxr;
uint32_t xmlOffset = 0; //-- current xml byte offset
uint32_t txOffset = 0; //-- current tx-file offset

//-- suppress gcc warnings of the form "ignoring return value of `fread', declared with attribute warn_unused_result [-Wunused-result]"
#pragma GCC diagnostic ignored "-Wunused-result"

/*======================================================================
 * Utils
 */

//--------------------------------------------------------------
// dump_text()
//  + escape text for printing in dump
//  + returns global output buffer index_txtbuf
#define ITBUFSIZE 256
char index_txtbuf[ITBUFSIZE];
char *dump_text(const char *buf, int len)
{
  int i,j;
  char *out = index_txtbuf;
  for (i=0,j=0; (len < 0 || i < len) && buf[i]; i++) {
    switch (buf[i]) {
    case '\\':
      out[j++] = '\\';
      out[j++] = '\\';
      break;
    case '\t':
      out[j++] = '\\';
      out[j++] = 't';
      break;
    case '\n':
      out[j++] = '\\';
      out[j++] = 'n';
      break;
    case '\r':
      out[j++] = '\\';
      out[j++] = 'r';
      break;
    default:
      out[j++] = buf[i];
      break;
    }
  }
  out[j++] = '\0';
  return out;
}

//--------------------------------------------------------------
void dump_header(FILE *f, const cxHeader *h, int argc, char **argv)
{
  int i;
  if (!f) return;
  fprintf(f, "%%%% cx-data dump generated by %s\n", prog);
  fprintf(f, "%%%% Package: %s version %s / %s\n", PACKAGE, PACKAGE_VERSION, PACKAGE_SVNID);
  fprintf(f, "%%%% Command-line: %s", argv[0]);
  for (i=1; i < argc; i++) { fprintf(f," '%s'", argv[i]); }
  fputc('\n', f);
  fprintf(f, "%%%%\n");
  fprintf(f, "%%%% @MAGIC=%s\n", dump_text(h->magic,-1));
  fprintf(f, "%%%% @VERSION=%s\n", dump_text(h->version,-1));
  fprintf(f, "%%%% @VERSION_MIN=%s\n", dump_text(h->version_min,-1));
  fprintf(f, "%%%%\n");
  fprintf(f, "%%%%$TYPE$\t$XML_OFFSET$\t$XML_LENGTH$\t$TX_OFFSET$\t$TX_LEN$\t$TEXT$?\t$ATTRS$?\n");
  fprintf(f, "%%%%======================================================================\n");
}

//--------------------------------------------------------------
uchar txbuf[ITBUFSIZE];
void dump_record(FILE *f, const cxStoredRecord *cxr, FILE *f_tx)
{
  if (!f) return;
  txbuf[0] = '\0';
  if (f_tx && cxr->tlen) {
    fseek(f_tx, txOffset, SEEK_SET);
    fread(txbuf, 1, cxr->tlen, f_tx);
    txbuf[cxr->tlen] = '\0';
  }
  fprintf(f, "%s\t%u\t%hhu\t%u\t%hhu\t%s",
	  cxTypeNames[(cxr->flags&cxfTypeMask)],
	  (uint)cxr->xoff, cxr->xlen,
	  (uint)txOffset, cxr->tlen,
	  dump_text((char*)txbuf,(int)cxr->tlen));

  //-- attrs
  if (cxr->flags&cxfHasAttrs) {
    switch (cxr->flags&cxfTypeMask) {
    case cxrChar: fprintf(f, "\tulx=%u uly=%u lrx=%u lry=%u", (uint)cxr->attrs[0], (uint)cxr->attrs[1], (uint)cxr->attrs[2], (uint)cxr->attrs[3]); break;
    case cxrPb:   fprintf(f, "\tfacs=#f%04u", (uint)cxr->attrs[0]); break;
    default: break;
    }
  }
  fputc('\n',f);
}

/*======================================================================
 * MAIN
 */
int main(int argc, char **argv)
{
  char *filename_cx = "-";
  char *filename_tx = NULL;
  char *filename_out = "-";
  FILE *f_cx = stdin;  //-- input cx-file (required)
  FILE *f_tx = NULL;   //-- input tx-file (optional)
  FILE *f_out = stdout;  //-- output tab-separated cx-file (optional)
  cxHeader hdr;

  //-- initialize: globals
  prog = file_basename(NULL,argv[0],"",-1,0);

  //-- command-line: usage
  if (argc <= 1) {
    fprintf(stderr, "(%s version %s / %s)\n", PACKAGE, PACKAGE_VERSION, PACKAGE_SVNID);
    fprintf(stderr, "Usage:\n");
    fprintf(stderr, " + %s CXFILE [TXFILE [OUTFILE]]\n", prog);
    fprintf(stderr, " + CXFILE : input character-index binary file; default=stdin\n");
    fprintf(stderr, " + OUTFILE: output raw cx-file dump; default=stdout\n");
    fprintf(stderr, " + \"-\" may be used in place of any filename to indicate standard (in|out)put\n");
    fprintf(stderr, " + \"\"  may be used in place of any output filename to discard output\n");
    exit(1);
  }
  //-- command-line: input: cx-file
  if (argc > 1) {
    filename_cx = argv[1];
    if ( strcmp(filename_cx,"-")!=0 && !(f_cx=fopen(filename_cx,"rb")) ) {
      fprintf(stderr, "%s: open failed for input cx-file `%s': %s\n", prog, filename_cx, strerror(errno));
      exit(1);
    }
  }
  //-- command-line: input: tx-file
  if (argc > 2) {
    filename_tx = argv[2];
    if (strcmp(filename_tx,"")==0) {
      f_tx = NULL;
    }
    else if ( strcmp(filename_tx,"-")==0 ) {
      f_tx = stdin;
    }
    else if ( !(f_tx=fopen(filename_tx,"rb")) ) {
      fprintf(stderr, "%s: open failed for input text-data file `%s': %s\n", prog, filename_tx, strerror(errno));
      exit(1);
    }
  }
  //-- command-line: output: cx-dump
  if (argc > 3) {
    filename_out = argv[3];
    if (strcmp(filename_out,"")==0) {
      f_out = NULL;
    }
    else if ( strcmp(filename_out,"-")==0 ) {
      f_out = stdout;
    }
    else if ( !(f_out=fopen(filename_out,"wb")) ) {
      fprintf(stderr, "%s: open failed for output cx-dump file `%s': %s\n", prog, filename_out, strerror(errno));
      exit(1);
    }
  }

  //-- get & dump header
  cx_get_header(f_cx, filename_cx, &hdr);
  dump_header(f_out, &hdr, argc, argv);
  if (!cx_check_header(&hdr,filename_cx)) exit(1);

  //-- churn cx-records
  while (f_cx && !feof(f_cx) && cx_get_record(f_cx, &cxr, xmlOffset) != cxrEOF) {
    dump_record(f_out, &cxr, f_tx);
    xmlOffset = cxr.xoff + cxr.xlen;
    txOffset += cxr.tlen;
  }

  //-- cleanup
  if (f_cx) fclose(f_cx);
  if (f_tx && f_tx != f_cx) fclose(f_tx);
  if (f_out) fclose(f_out);

  return 0;
}
