## -*- Mode: CPerl -*-

## File: DTA::TokWrap::Processor::tokenize::dummy.pm
## Author: Bryan Jurish <moocow@cpan.org>
## Descript: DTA tokenizer wrappers: dtatw-tokenize-dummy

package DTA::TokWrap::Processor::tokenize::dummy;

use DTA::TokWrap::Version;
use DTA::TokWrap::Base;
use DTA::TokWrap::Utils qw(:progs :slurp :time);
use DTA::TokWrap::Processor;
use DTA::TokWrap::Processor::tokenize;

use Carp;
use strict;

##==============================================================================
## Constants
##==============================================================================
our @ISA = qw(DTA::TokWrap::Processor::tokenize);

##==============================================================================
## Constructors etc.
##==============================================================================

## $td = CLASS_OR_OBJ->new(%args)
##  + %args:
##    tokenize => $path_to_dtatw_tokenize, ##-- default: search
##    inplace  => $bool,                   ##-- prefer in-place programs for search?
sub new { return $_[0]->DTA::TokWrap::Processor::new(@_[1..$#_]); }

## %defaults = CLASS->defaults()
sub defaults {
  my $that = shift;
  return (
	  #$that->SUPER::defaults(), ##-- block inheritance from DTA::TokWrap::Processor::tokenize
	  $that->DTA::TokWrap::Processor::defaults(),

	  tokenize=>undef,
	  inplace=>1,
	 );
}

## $td = $td->init()
sub init {
  my $td = shift;

  ##-- search for tokenizer program
  if (!defined($td->{tokenize})) {
    $td->{tokenize} = path_prog('dtatw-tokenize-dummy',
				prepend=>($td->{inplace} ? ['.','../src'] : undef),
				warnsub=>sub {$td->logconfess(@_)},
			       );
  }

  return $td;
}

##==============================================================================
## Methods
##==============================================================================

## $doc_or_undef = $CLASS_OR_OBJECT->tokenize($doc)
## + $doc is a DTA::TokWrap::Document object
## + %$doc keys:
##    txtfile => $txtfile,  ##-- (input) serialized text file
##    tokdata => $tokdata,  ##-- (output) tokenizer output data (string)
##    ntoks => $nTokens,    ##-- (output) number of output tokens (regex hack)
##    tokenize_stamp0 => $f, ##-- (output) timestamp of operation begin
##    tokenize_stamp  => $f, ##-- (output) timestamp of operation end
##    tokdata_stamp => $f,   ##-- (output) timestamp of operation end
## + may implicitly call $doc->mkbx() and/or $doc->saveTxtFile()
sub tokenize {
  my ($td,$doc) = @_;
  $doc->setLogContext();

  ##-- log, stamp
  $td->vlog($td->{traceLevel},"tokenize()");
  $doc->{tokenize_stamp0} = timestamp();

  ##-- sanity check(s)
  $td = $td->new if (!ref($td));
  $td->logconfess("tokenize(): no dtatw-tokenize-dummy program")
    if (!$td->{tokenize});
  $td->logconfess("tokenize(): no .txt file defined")
    if (!defined($doc->{txtfile}));
  $td->logconfess("tokenize(): .txt file '$doc->{txtfile}' not readable")
    if (!-r $doc->{txtfile});

  ##-- run program
  $doc->{tokdata0} = '';
  my $cmdfh = opencmd("'$td->{tokenize}' '$doc->{txtfile}'|")
    or $td->logconfess("tokenize(): open failed for pipe ('$td->{tokenize}' '$doc->{txtfile}' |): $!");
  slurp_fh($cmdfh, \$doc->{tokdata0});
  $cmdfh->close();

  ##-- finalize
  $doc->{ntoks} = $td->nTokens(\$doc->{tokdata0});
  $doc->{tokfile0_stamp} = $doc->{tokenize0_stamp} = $doc->{tokdata0_stamp} = timestamp(); ##-- stamp
  return $doc;
}


1; ##-- be happy

__END__
##========================================================================
## POD DOCUMENTATION, auto-generated by podextract.perl, edited

##========================================================================
## NAME
=pod

=head1 NAME

DTA::TokWrap::Processor::tokenize::dummy - DTA tokenizer wrappers: dtatw-tokenize-dummy

=cut

##========================================================================
## SYNOPSIS
=pod

=head1 SYNOPSIS

 use DTA::TokWrap::Processor::tokenize::dummy;
 
 $td = DTA::TokWrap::Processor::tokenize::dummy->new(%args);
 $doc_or_undef = $td->tokenize($doc);

=cut

##========================================================================
## DESCRIPTION
=pod

=head1 DESCRIPTION

DTA::TokWrap::Processor::tokenize::dummy provides a package-local
alternative to the "official" low-level tokenizer class
L<DTA::TokWrap::Processor::tokenize|DTA::TokWrap::Processor::tokenize>.

Most users should use the high-level
L<DTA::TokWrap|DTA::TokWrap> wrapper class
instead of using this module directly.

=cut

##----------------------------------------------------------------
## DESCRIPTION: DTA::TokWrap::Processor::tokenize::dummy: Constants
=pod

=head2 Constants

=over 4

=item @ISA

DTA::TokWrap::Processor::tokenize::dummy
inherits from
L<DTA::TokWrap::Processor::tokenize|DTA::TokWrap::Processor::tokenize>.

=back

=cut

##----------------------------------------------------------------
## DESCRIPTION: DTA::TokWrap::Processor::tokenize::dummy: Constructors etc.
=pod

=head2 Constructors etc.

=over 4

=item new

 $td = $CLASS_OR_OBJ->new(%args);

Constructor.

%args, %$td:

 tokenize => $path_to_dtatw_tokenize, ##-- default: search
 inplace  => $bool,                   ##-- prefer in-place programs for search?

=item defaults

 %defaults = $CLASS->defaults();

Static class-dependent defaults.

=item init

 $td = $td->init();

Dynamic object-dependent defaults.

=back

=cut

##----------------------------------------------------------------
## DESCRIPTION: DTA::TokWrap::Processor::tokenize::dummy: Methods
=pod

=head2 Methods

=over 4

=item tokenize

 $doc_or_undef = $CLASS_OR_OBJECT->tokenize($doc);

See L<DTA::TokWrap::Processor::tokenize::tokenize()|DTA::TokWrap::Processor::tokenize/tokenize>.

=back

=cut

##========================================================================
## END POD DOCUMENTATION, auto-generated by podextract.perl

##======================================================================
## See Also
##======================================================================

=pod

=head1 SEE ALSO

L<DTA::TokWrap::Intro(3pm)|DTA::TokWrap::Intro>,
L<dta-tokwrap.perl(1)|dta-tokwrap.perl>,
...

=cut

##======================================================================
## See Also
##======================================================================

=pod

=head1 SEE ALSO

L<DTA::TokWrap::Intro(3pm)|DTA::TokWrap::Intro>,
L<dta-tokwrap.perl(1)|dta-tokwrap.perl>,
...

=cut

##======================================================================
## Footer
##======================================================================

=pod

=head1 AUTHOR

Bryan Jurish E<lt>moocow@cpan.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2009-2018 by Bryan Jurish

This package is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.14.2 or,
at your option, any later version of Perl 5 you may have available.

=cut


