# Using the BitPay Perl Client Library
## Prerequisites
You must have a BitPay merchant account to use this library.  It's free to [sign-up for a BitPay merchant account](https://bitpay.com/start).

Once you have a BitPay merchant account, you will need [a working BitPay Access Token](/api/getting-access.html) – this can be done either [via the library](#pairing) or manually in [the BitPay Dashboard](https://bitpay.com/tokens).

## Quick Start
### Installation

```bash
cpanm Business::OnlinePayment::BitPay::Client
```

### Configuration
The bitpay client creates a cryptographically secure connection to your server by pairing an API code with keys generated by the library. The client must be initialized with pre-existing keys passed in as a pem string. The Business::OnlinePayment::BitPay::KeyUtils package has a method for creating keys. See [the key utilities documentation](https://github.com/bitpay/bitpay-perl-keyutils/blob/master/GUIDE.md) for examples.

## Pairing
Most calls to the BitPay REST API require that your client is paired with the bitpay.com server.  To pair with bitpay.com you need to have an approved merchant account.

Your client can be paired via the `pos` (point-of-sale) or `merchant` facade (or both).  The `pos` facade allows for invoices to be created.  The `merchant` facade has broader privileges to view all invoices, bills, and ledger entries, as well as to issue refunds.  Consider the level of access required when you pair your client.

### A quick note on keys

The BitPay client package includes the BitPay KeyUtilities package, which can be used to generate new public private key pairs which it returns in PEM format. However, there are no methods which save the keys anywhere, so it is your responsibility to store the PEM file somewhere secure.

### BitPay authentication

BitPay authentication depends on four parts:

1. An account on our servers.
1. A token shared between the client and the server.
1. A public key, shared between the client and the server.
1. A private key, held exclusively by the client.

In order to complete authentication, you have to associate your private key with a token, and associate that token with an account. Once this authentication is complete, as long as you have the private key, you never have to authenticate again. The token you created will always be associated with that private key, so any time you create a new bitpay client object with that key, it is authenticated with BitPay. This is true whether you use the perl-client, python client, or no client at all, the key is the important thing.

There are two ways to authenticate, from the client side or the server side. The Perl Client supports both.

To pair from the server side, you log in to the BitPay server, navigate to dashboard/merchant/api-tokens, and create a new token. This creates a new token, which is associated with your account. It is not associated with a key, so we provide a pairing code that you can use as a one time secret to associate the token with a key. From the client side, you can use the `client->pair_pos_client(<pairing_code>)` subroutine to associate the server token with a key held by the client.

To pair from the client side, you use the client to call the /tokens endpoint on the server with no parameters, or use the `client->pair_client(facade => < 'pos' | 'merchant' > )` subroutine. This creates a token on the server and associates that token with a public key. What it doesn't do is associate that token to an account (because we don't know what account to associate with). This call returns a token map including a pairing code, which is a one time secret that allows you to find the token you just created. In order to associate the token with an account, you log in to the BitPay server, and use the dashboard/merchant/api-tokens interface to associate the token with a specific account. An example of client side pairing is shown below.

```perl
    my $pem = BitPay::KeyUtils::bpGeneratePem(); # remember, this must be saved in order to use it again!
    my $uri = "https://test.bitpay.com" # or https://bitpay.com for live use
    my %options = ("pem" => $pem, "apiUri" => $uri);
    my $client = Business::OnlinePayment::BitPay::Client->new(%options);
    my @data = $client->pair_client(facade => 'pos');
    $pairingCode = shift(shift(@data))->{'pairingCode'};
    print $pairingCode;
```

As described above, using the value from the `pairingCode` element, visit https://test.bitpay.com/api-tokens and search to register for the appropriate facade. That client is now paired. As previously mentioned, you **must save the pem string** you generated in order to use the client again.

## General Usage

### Initialize the client

```perl
open FILE, "bitpay.pem" or die $!;
my @lines = <FILE>;
$pem = join("", @lines);
my $pem = BitPay::KeyUtils::bpGeneratePem(); # remember, this must be saved in order to use it again!
my $uri = "https://test.bitpay.com" # or https://bitpay.com for live use
my %options = ("pem" => $pem, "apiUri" => $uri);
my $client = Business::OnlinePayment::BitPay::Client->new(%options);
```

### Create the invoice
```perl
my $price = 100.02;
my $currency = "USD";
my $invoice = $client.create_invoice (price => $price, currency: $currency);
```

With invoice creation, `price` and `currency` are the only required fields. If you are sending a customer from your website to make a purchase, setting `redirectURL` will redirect the customer to your website when the invoice is paid.

Response will be a hash with information on your newly created invoice. Send your customer to the `url` to complete payment:

```javascript
{
  "url": "https://bitpay.com/invoice?id=NKaqMuZWy3BAcP77RdkEEv",
  "paymentUrls": {
    "BIP21": "bitcoin:mvYRECDxKPaPHnjNz9ZxiTpbx29xYNoRy4?amount=0.3745",
    "BIP72": "bitcoin:mvYRECDxKPaPHnjNz9ZxiTpbx29xYNoRy4?amount=0.3745&r=https://bitpay.com/i/NKaqMuZWy3BAcP77RdkEEv",
    "BIP72b": "bitcoin:?r=https://bitpay.com/i/NKaqMuZWy3BAcP77RdkEEv",
    "BIP73": "https://bitpay.com/i/NKaqMuZWy3BAcP77RdkEEv"
  },
  "status": "new",
  "btcPrice": "0.3745",
  "btcDue": "0.3745",
  "price": 148,
  "currency": "USD",
  "exRates": {
    "USD": 395.20000000000005
  },
  "invoiceTime": 1415987168612,
  "expirationTime": 1415988068612,
  "currentTime": 1415987168629,
  "guid": "438e8237-fff1-483c-81b4-dc7dba28922a",
  "id": "NKaqMuZWy3BAcP77RdkEEv",
  "transactions": [

  ],
  "btcPaid": "0.0000",
  "rate": 395.2,
  "exceptionStatus": false,
  "token": "9kZgUXFb5AC6qMuLaMpP9WopbM8X2UjMhkphKKdaprRbSKgUJNE6JNTX8bGsmgxKKv",
  "buyer": {
  }
}
```

There are many options available when creating invoices, which are listed in the [BitPay API documentation](https://bitpay.com/bitcoin-payment-gateway-api).

### Get invoice status
The perl library provides a method for fetching an existing invoice:

```perl
$client->get_invoice(id => 'PvVhgBfA7wKPWhuVC24rJo', public => 1);
```

The "public" argument allows non-authenticated clients to retrieve invoices. This cannot presently be omitted but this will change with the introduction of refunds in v2.4.

### Make a HTTP request directly against the REST API

For API tasks which lack a dedicated library method, either the `client-get()` or `client->post()` subroutines will automatically apply the proper cryptographic parameters to a request. The post method is looking for an endpoint and a hash of params to be passed in. The `create_invoice` subroutine is a good example of how to set up a post request. Similary, `get_invoice` or `retrieve_pos_or_merchant_tokens_from_server` are examples of usage for the `get` method.

## Testnet Usage

During development and testing, take advantage of the [Bitcoin TestNet](https://en.bitcoin.it/wiki/Testnet) by passing the testnet URI when creating a client:

```perl
my $uri = "https://test.bitpay.com" # or https://bitpay.com for live use
my %options = ("pem" => $pem, "apiUri" => $uri);
my $client = Business::OnlinePayment::BitPay::Client->new(%options);
```

Note that in order to pair with testnet, you will need a pairing code from test.bitpay.com.

## API Documentation

API Documentation is available on the [BitPay site](https://bitpay.com/api).
