package Sisimai::Reason::MesgTooBig;
use feature ':5.10';
use strict;
use warnings;

sub text  { 'mesgtoobig' }
sub match {
    my $class = shift;
    my $argvs = shift // return undef;
    my $regex = qr{(?>
         exceeded[ ]maximum[ ]inbound[ ]message[ ]size
        |max[ ]message[ ]size[ ]exceeded
        |message[ ](?:
             file[ ]too[ ]big
            |length[ ]exceeds[ ]administrative[ ]limit
            |size[ ]exceeds[ ](?:
                 fixed[ ]limit
                |fixed[ ]maximum[ ]message[ ]size
                |maximum[ ]value
                )
            |too[ ]big
            |too[ ]large[ ]for[ ]this[ ].+
            )
        |size[ ]limit
        )
    }ix;

    return 1 if $argvs =~ $regex;
    return 0;
}

sub true {
    # @Description  The message size is too big for the remote host
    # @Param <obj>  (Sisimai::Data) Object
    # @Return       (Integer) 1 = is too big message size
    #               (Integer) 0 = is not big
    # @See          http://www.ietf.org/rfc/rfc2822.txt
    my $class = shift;
    my $argvs = shift // return undef;

    return undef unless ref $argvs eq 'Sisimai::Data';
    return 1 if $argvs->reason eq __PACKAGE__->text;

    require Sisimai::RFC3463;
    my $statuscode = $argvs->deliverystatus // '';
    my $reasontext = __PACKAGE__->text;
    my $tempreason = '';
    my $diagnostic = '';
    my $v = 0;

    $diagnostic = $argvs->diagnosticcode // '';
    $tempreason = Sisimai::RFC3463->reason( $statuscode ) if $statuscode;

    if( $tempreason eq $reasontext ) {
        # Delivery status code points "mesgtoobig".
        # Status: 5.3.4
        # Diagnostic-Code: SMTP; 552 5.3.4 Error: message file too big
        $v = 1;

    } else {
        if( $tempreason eq 'exceedlimit' || $statuscode eq '5.2.3' ) {
            #  5.2.3   Message length exceeds administrative limit
            $v = 0;

        } else {
            # Check the value of Diagnosic-Code: header with patterns
            $v = 1 if __PACKAGE__->match( $diagnostic );
        }
    }

    return $v;
}

1;
__END__

=encoding utf-8

=head1 NAME

Sisimai::Reason::MesgTooBig - Bounce reason is C<mesgtoobig> or not.

=head1 SYNOPSIS

    use Sisimai::Reason::MesgTooBig;
    print Sisimai::Reason::MesgTooBig->match('400 Message too big');   # 1

=head1 DESCRIPTION

Sisimai::Reason::MesgTooBig checks the bounce reason is C<mesgtoobig> or not.
This class is called only Sisimai::Reason class.

=head1 CLASS METHODS

=head2 C<B<text()>>

C<text()> returns string: C<mesgtoobig>.

    print Sisimai::Reason::MesgTooBig->text;  # mesgtoobig

=head2 C<B<match( I<string> )>>

C<match()> returns 1 if the argument matched with patterns defined in this class.

    print Sisimai::Reason::MesgTooBig->match('400 Message too big');   # 1

=head2 C<B<true( I<Sisimai::Data> )>>

C<true()> returns 1 if the bounce reason is C<mesgtoobig>. The argument must be
Sisimai::Data object and this method is called only from Sisimai::Reason class.

=head1 AUTHOR

azumakuniyuki

=head1 COPYRIGHT

Copyright (C) 2014-2015 azumakuniyuki E<lt>perl.org@azumakuniyuki.orgE<gt>,
All Rights Reserved.

=head1 LICENSE

This software is distributed under The BSD 2-Clause License.

=cut
