package Aspect::Advice::AfterReturning;

use strict;
use warnings;

# Added by eilara as hack around caller() core dump
# NOTE: Now we've switched to Sub::Uplevel can this be removed? --ADAMK
use Carp::Heavy                   (); 
use Carp                          ();
use Sub::Uplevel                  ();
use Aspect::Hook                  ();
use Aspect::Advice                ();
use Aspect::Point::AfterReturning ();

our $VERSION = '0.94';
our @ISA     = 'Aspect::Advice';

# NOTE: To simplify debugging of the generated code, all injected string
# fragments will be defined in $UPPERCASE, and all lexical variables to be
# accessed via the closure will be in $lowercase.
sub _install {
	my $self     = shift;
	my $pointcut = $self->pointcut;
	my $code     = $self->code;
	my $lexical  = $self->lexical;

	# Get the curried version of the pointcut we will use for the
	# runtime checks instead of the original.
	# Because $MATCH_RUN is used in boolean conditionals, if there
	# is nothing to do the compiler will optimise away the code entirely.
	my $curried   = $pointcut->match_curry;
	my $compiled  = $curried ? $curried->compiled_runtime : undef;
	my $MATCH_RUN = $compiled ? '$compiled->()' : 1;

	# When an aspect falls out of scope, we don't attempt to remove
	# the generated hook code, because it might (for reasons potentially
	# outside our control) have been recursively hooked several times
	# by both Aspect and other modules.
	# Instead, we store an "out of scope" flag that is used to shortcut
	# past the hook as quickely as possible.
	# This flag is shared between all the generated hooks for each
	# installed Aspect.
	# If the advice is going to last lexical then we don't need to
	# check or use the $out_of_scope variable.
	my $out_of_scope   = undef;
	my $MATCH_DISABLED = $lexical ? '$out_of_scope' : '0';

	# Find all pointcuts that are statically matched
	# wrap the method with advice code and install the wrapper
	foreach my $name ( $pointcut->match_all ) {
		my $NAME = $name; # For completeness

		no strict 'refs';
		my $original = *$name{CODE};
		unless ( $original ) {
			Carp::croak("Can't wrap non-existent subroutine ", $name);
		}

		# Any way to set prototypes other than eval?
		my $PROTOTYPE = prototype($original);
		   $PROTOTYPE = defined($PROTOTYPE) ? "($PROTOTYPE)" : '';

		# Generate the new function
		no warnings 'redefine';
		eval <<"END_PERL"; die $@ if $@;
		package Aspect::Hook;

		*$NAME = sub $PROTOTYPE {
			# Is this a lexically scoped hook that has finished
			goto &\$original if $MATCH_DISABLED;

			my \$wantarray = wantarray;
			if ( \$wantarray ) {
				my \$return = [
					Sub::Uplevel::uplevel(
						1, \$original, \@_,
					)
				];
				local \$_ = bless {
					type         => 'after_returning',
					sub_name     => \$name,
					wantarray    => \$wantarray,
					params       => \\\@_,
					return_value => \$return,
					exception    => undef,
					pointcut     => \$pointcut,
					original     => \$original,
				}, 'Aspect::Point::AfterReturning';
				return \@\$return unless $MATCH_RUN;

				# Execute the advice code
				() = &\$code(\$_);

				# Get the (potentially) modified return value
				return \@{\$_->{return_value}};
			}

			if ( defined \$wantarray ) {
				my \$return = Sub::Uplevel::uplevel(
					1, \$original, \@_,
				);
				local \$_ = bless {
					type         => 'after_returning',
					sub_name     => \$name,
					wantarray    => \$wantarray,
					params       => \\\@_,
					return_value => \$return,
					exception    => undef,
					pointcut     => \$pointcut,
					original     => \$original,
				}, 'Aspect::Point::AfterReturning';
				return \$return unless $MATCH_RUN;

				# Execute the advice code
				my \$dummy = &\$code(\$_);
				return \$_->{return_value};

			} else {
				Sub::Uplevel::uplevel(
					1, \$original, \@_,
				);
				local \$_ = bless {
					type         => 'after_returning',
					sub_name     => \$name,
					wantarray    => \$wantarray,
					params       => \\\@_,
					return_value => undef,
					exception    => undef,
					pointcut     => \$pointcut,
					original     => \$original,
				}, 'Aspect::Point::AfterReturning';
				return unless $MATCH_RUN;

				# Execute the advice code
				&\$code(\$_);
				return;
			}
		};
END_PERL
	}

	# If this will run lexical we don't need a descoping hook
	return unless $lexical;

	# Return the lexical descoping hook.
	# This MUST be stored and run at DESTROY-time by the
	# parent object calling _install. This is less bullet-proof
	# than the DESTROY-time self-executing blessed coderef
	return sub { $out_of_scope = 1 };
}

1;

__END__

=pod

=head1 NAME

Aspect::Advice::AfterReturning - Execute code when a function returns
without throwing an exception.

=head1 SYNOPSIS

  use Aspect;
  
  after_returning {
      # Trace all normally returning calls to your module
      print STDERR "Called my function " . $_->sub_name . "\n";
  
      # Throw an exception if foo() returns 'bar'
      if ( $_->short_sub_name eq 'foo' and $_->return_value eq 'bar' ) {
          $_->exception("Missing or invalid arguments to foo()");
      }
  
  } call qr/^ MyModule::\w+ $/

=head1 DESCRIPTION

The C<after_returning> advice type is used to execute code after a function
has returned, but C<WITHOUT> trapping exceptions (they will propogate
upwards normally).

As well as creating side effects that run after the main code, the
C<after_returning> advice type is particularly useful for selectively
altering the return value of a function or API-shifting functions which
signal errors by returning C<undef> so they throw exceptions instead.

=head1 AUTHORS

Adam Kennedy E<lt>adamk@cpan.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2010 Adam Kennedy.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=cut
