#define PERL_NO_GET_CONTEXT
#include "EXTERN.h"
#include "perl.h"
#include "XSUB.h"

#include "ppport.h"

#include <secp256k1.h>

struct _blob32 {
  unsigned char data[32];
};
typedef unsigned char  *secp256k1_blob32_cast;
typedef struct _blob32 secp256k1_seckey;
typedef struct _blob32 secp256k1_random;
typedef struct _blob32 secp256k1_tweak;
typedef struct _blob32 secp256k1_hash;
typedef secp256k1_context *Crypt__secp256k1;

typedef void (* secp256k1_error_function) (const char *, void *);
typedef void (* secp256k1_illegal_function) (const char *, void *);


/* Callback rewrappers */

int
crypt__secp256k1__nonce_function(nonce32, msg32, key32, algo16, data, attempt)
    unsigned char       * nonce32;
    const unsigned char * msg32;
    const unsigned char * key32;
    const unsigned char * algo16;
    void                * data;
    unsigned int          attempt;
{
  dSP;
  int  count;
  SV  *err_tmp;
  int  len;
  int  r = 0;

  ENTER;
  SAVETMPS;

  PUSHMARK(SP);
  EXTEND(SP, 4); /* Pass it msg key [algo [attempt]] */
  len = 32;
  PUSHs(sv_2mortal(newSVpv(msg32, len)));
  /* assert len == 32? */
  PUSHs(sv_2mortal(newSVpv(key32, len)));
  len = 16;
  PUSHs(sv_2mortal(newSVpv(algo16, len)));
  PUSHs(sv_2mortal(newSViv(attempt)));
  PUTBACK;

  /* data contains the perl coderef */
  count = call_sv((SV *)data, G_SCALAR | G_EVAL);
  /* assert count == 1 */

  /* Check $@ */
  SPAGAIN;
  err_tmp = ERRSV;
  if (SvTRUE(err_tmp)) {
    printf ("Died: %s\n", SvPV_nolen(err_tmp));
    POPs;
    r = 0;
  } else {
    STRLEN   _nonce_len;
    memcpy(nonce32, SvPV(POPs, _nonce_len), 32);
    r = 1;
  }
  PUTBACK;

  FREETMPS;
  LEAVE;

  return r;
}

void
crypt__secp256k1__illegal_function(message, data)
    const char * message;
    void       * data;
{
  int count;
  ENTER;
  SAVETMPS;
  count = call_sv((SV *)data, G_DISCARD);
  /* assert count == 1 */
  FREETMPS;
  LEAVE;
}

void
crypt__secp256k1__error_function(message, data)
    const char * message;
    void       * data;
{
  int count;
  ENTER;
  SAVETMPS;
  count = call_sv((SV *)data, G_DISCARD);
  /* assert count == 1 */
  FREETMPS;
  LEAVE;
}

/* Context methods */
MODULE = Crypt::secp256k1  PACKAGE = Crypt::secp256k1  PREFIX = secp256k1_

PROTOTYPES: ENABLE

int
secp256k1_SECP256K1_CONTEXT_NONE()
  CODE:
    RETVAL = SECP256K1_CONTEXT_NONE;
  OUTPUT:
    RETVAL

int
secp256k1_SECP256K1_CONTEXT_SIGN()
  CODE:
    RETVAL = SECP256K1_CONTEXT_SIGN;
  OUTPUT:
    RETVAL

int
secp256k1_SECP256K1_CONTEXT_VERIFY()
  CODE:
    RETVAL = SECP256K1_CONTEXT_VERIFY;
  OUTPUT:
    RETVAL

Crypt::secp256k1
secp256k1_new(package, ...)
    char * package
  PREINIT:
    unsigned int flags = SECP256K1_CONTEXT_NONE;
  CODE:
    if (items >= 2) {
      /* If a number: */
      flags = (unsigned int) SvUV(ST(1));
    }
    RETVAL = secp256k1_context_create(flags);
    /* TODO: [optionally?] call randomize */
  OUTPUT:
    RETVAL

Crypt::secp256k1
secp256k1_clone(ctx)
    Crypt::secp256k1 ctx
  CODE:
    RETVAL = secp256k1_context_clone(ctx);
    /* TODO: [optionally?] call randomize */
  OUTPUT:
    RETVAL

SV *
DESTROY(ctx)
    Crypt::secp256k1 ctx
  CODE:
    /* Rename secp256k1_DESTROY? */
    secp256k1_context_destroy(ctx);

int
secp256k1_randomize(ctx, seed)
    Crypt::secp256k1  ctx
    secp256k1_random &seed
  CODE:
    RETVAL = secp256k1_randomize(ctx, (const secp256k1_blob32_cast)&seed);
    /* croak like the others? */
  OUTPUT:
    RETVAL

void
secp256k1_set_error_callback(ctx, perl_func)
    Crypt::secp256k1         ctx
    secp256k1_error_function perl_func
  CODE:
    secp256k1_context_set_error_callback(ctx,
                                           crypt__secp256k1__error_function,
                                           (void *)perl_func);

void
secp256k1_set_illegal_callback(ctx, perl_func)
     Crypt::secp256k1          ctx
    secp256k1_illegal_function perl_func
  CODE:
    secp256k1_context_set_illegal_callback(ctx,
                                           crypt__secp256k1__illegal_function,
                                           (void *)perl_func);


# Key methods 

MODULE = Crypt::secp256k1  PACKAGE = Crypt::secp256k1  PREFIX = secp256k1_ec_

SV *
secp256k1_ec_privkey_negate(ctx, seckey)
    Crypt::secp256k1 ctx
    secp256k1_seckey seckey
  CODE:
    if (!(secp256k1_ec_privkey_negate(ctx, (secp256k1_blob32_cast)&seckey)))
      Perl_croak("Error negating seckey");
    RETVAL = newSVpv((const char * const)&seckey, sizeof(seckey));
  OUTPUT:
    RETVAL

SV *
secp256k1_ec_privkey_tweak_add(ctx, seckey, tweak)
    Crypt::secp256k1 ctx
    secp256k1_seckey seckey
    secp256k1_tweak  tweak
  CODE:
    if (!(secp256k1_ec_privkey_tweak_add(ctx,
                                        (secp256k1_blob32_cast) &seckey,
                                        (const secp256k1_blob32_cast)&tweak)))
      Perl_croak("Error adding to seckey");
    RETVAL = newSVpv((const char * const)&seckey, sizeof(seckey));
  OUTPUT:
    RETVAL

SV *
secp256k1_ec_privkey_tweak_mul(ctx, seckey, tweak)
    Crypt::secp256k1 ctx
    secp256k1_seckey seckey
    secp256k1_tweak  tweak
  CODE:
   if (!(secp256k1_ec_privkey_tweak_mul(ctx,
                                        (secp256k1_blob32_cast) &seckey,
                                        (const secp256k1_blob32_cast)&tweak)))
      Perl_croak("Error multiplying seckey");
    RETVAL = newSVpv((const char * const)&seckey, sizeof(seckey));
  OUTPUT:
    RETVAL

SV *
secp256k1_ec_pubkey_combine(ctx, ...)
    Crypt::secp256k1 ctx
  INIT:
    secp256k1_pubkey **in;
    int                i;
    secp256k1_pubkey   out;
  CODE:
    if (items < 2)
      Perl_croak("Error combining pubkeys");
    Newx(in, items - 1, secp256k1_pubkey *);
    for (i = 0; i <= (items - 1); i++) {
      STRLEN _pubkey_len;
      memcpy(in + i, SvPV(ST(i + 2), _pubkey_len), 64);
    }
    if (!(secp256k1_ec_pubkey_combine(ctx,
                                      &out,
                                      (const secp256k1_pubkey * const *)in,
                                      items - 1)))
      Perl_croak("Error combining pubkeys");
    RETVAL = newSVpv((const char * const)&out, sizeof(out));
  OUTPUT:
    RETVAL

SV *
secp256k1_ec_pubkey_create(ctx, seckey)
    Crypt::secp256k1 ctx
    secp256k1_seckey seckey
  INIT:
    secp256k1_pubkey pubkey;
  CODE:
    if (!(secp256k1_ec_pubkey_create(ctx, &pubkey, (const secp256k1_blob32_cast)&seckey)))
      Perl_croak("Error creating pubkey");
    RETVAL = newSVpv((const char * const)&pubkey, sizeof(pubkey));
  OUTPUT:
    RETVAL

SV *
secp256k1_ec_pubkey_negate(ctx, pubkey)
    Crypt::secp256k1 ctx
    secp256k1_pubkey pubkey
  CODE:
    if (!(secp256k1_ec_pubkey_negate(ctx, &pubkey)))
      Perl_croak("Error negating pubkey");
    RETVAL = newSVpv((const char * const)&pubkey, sizeof(pubkey));
  OUTPUT:
    RETVAL

SV *
secp256k1_ec_pubkey_parse(ctx, input, size_t length(input))
    Crypt::secp256k1      ctx
    const unsigned char * input
  INIT:
    secp256k1_pubkey      pubkey;
  CODE:
    if (!(secp256k1_ec_pubkey_parse(ctx, &pubkey, input, XSauto_length_of_input)))
      Perl_croak("Error negating pubkey");
    RETVAL = newSVpv((const char * const)&pubkey, sizeof(pubkey));
  OUTPUT:
    RETVAL

unsigned char *
secp256k1_ec_pubkey_serialize(ctx, pubkey, flags)
    Crypt::secp256k1 ctx
    secp256k1_pubkey pubkey
    unsigned int     flags
  PPCODE:
    char                      * output;
    size_t                      outputlen = 64;
    Newx(output, outputlen, char); /* random number */
    if (!(secp256k1_ecdsa_pubkey_serialize(ctx, output, &outputlen, &pubkey, flags)))
      Perl_croak("Error serializing pubkey");
    sv_setpvn(ST(1), (char *)&output, outputlen);
    SvSETMAGIC(ST(1));
    Safefree(output);
    XSRETURN(1);

SV *
secp256k1_ec_pubkey_tweak_add(ctx, pubkey, tweak)
    Crypt::secp256k1 ctx
    secp256k1_pubkey pubkey
    secp256k1_tweak  tweak
  CODE:
   if (!(secp256k1_ec_pubkey_tweak_add(ctx, &pubkey, (const secp256k1_blob32_cast)&tweak)))
      Perl_croak("Error adding to pubkey");
    RETVAL = newSVpv((const char * const)&pubkey, sizeof(pubkey));
  OUTPUT:
    RETVAL

SV *
secp256k1_ec_pubkey_tweak_mul(ctx, pubkey, tweak)
    Crypt::secp256k1 ctx
    secp256k1_pubkey pubkey
    secp256k1_tweak  tweak
  CODE:
   if (!(secp256k1_ec_pubkey_tweak_mul(ctx, &pubkey, (const secp256k1_blob32_cast)&tweak)))
      Perl_croak("Error multiplying pubkey");
    RETVAL = newSVpv((const char * const)&pubkey, sizeof(pubkey));
  OUTPUT:
    RETVAL

int
secp256k1_ec_seckey_verify(ctx, seckey)
    Crypt::secp256k1 ctx
    secp256k1_seckey seckey
  CODE:
    RETVAL = secp256k1_ec_seckey_verify(ctx, (const secp256k1_blob32_cast)&seckey);
    /* croak like the others? */
  OUTPUT:
    RETVAL


# Signature methods

MODULE = Crypt::secp256k1  PACKAGE = Crypt::secp256k1  PREFIX = secp256k1_ecdsa_

SV *
secp256k1_ecdsa_sign(ctx, seckey, msg, perl_nonce = NULL)
    Crypt::secp256k1            ctx
    secp256k1_seckey            seckey
    secp256k1_hash              msg
    SV                        * perl_nonce
  INIT:
    secp256k1_ecdsa_signature   sig;
    secp256k1_nonce_function    nonce = NULL;
    const void                * nonce_data = NULL;
  CODE:
    if (perl_nonce) {
      nonce = (void *)crypt__secp256k1__nonce_function;
      nonce_data = perl_nonce;
    }
    secp256k1_ecdsa_sign(ctx,
                         &sig,
                         (const unsigned char *)&msg,
                         (const unsigned char *)&seckey,
                         nonce,
                         nonce_data);
    RETVAL = newSVpv((const char * const)&sig, sizeof(sig));
  OUTPUT:
    RETVAL

SV *
secp256k1_ecdsa_signature_normalize(ctx, in)
    Crypt::secp256k1          ctx
    secp256k1_ecdsa_signature in
  INIT:
    int r;
    secp256k1_ecdsa_signature out;
  CODE:
    r = secp256k1_ecdsa_signature_normalize(ctx, &out, (const secp256k1_ecdsa_signature *)&in);
    if (!r) {
      /* was already normalised */
    } else {
      /* was not normalised */
    }
    RETVAL = newSVpv((const char * const)&out, sizeof(out));
  OUTPUT:
    RETVAL


SV *
secp256k1_ecdsa_signature_parse_compact(ctx, input)
    Crypt::secp256k1          ctx
    secp256k1_ecdsa_signature input; /* not really this type just looks like it */
  INIT:
    secp256k1_ecdsa_signature sig;
  CODE:
    if (!(secp256k1_ecdsa_signature_parse_compact(ctx,
                                                  &sig,
                                                  (const unsigned char *)&input)))
      Perl_croak("Error parsing signature");
    RETVAL = newSVpv((const char * const)&sig, sizeof(sig));
  OUTPUT:
    RETVAL

SV *
secp256k1_ecdsa_signature_parse_der(ctx, input, size_t length(input))
    Crypt::secp256k1          ctx
    secp256k1_ecdsa_signature input; /* not really this type just looks like it */
  INIT:
    secp256k1_ecdsa_signature sig;
  CODE:
    if (!(secp256k1_ec_signature_parse_der(ctx,
                                           &sig,
                                           (const unsigned char *)&input,
                                           XSauto_length_of_input)))
      Perl_croak("Error parsing DER signature");
    RETVAL = newSVpv((const char * const)&sig, sizeof(sig));
  OUTPUT:
    RETVAL

SV *
secp256k1_ecdsa_signature_serialize_compact(ctx, sig)
    Crypt::secp256k1          ctx
    secp256k1_ecdsa_signature sig
  INIT:
    secp256k1_ecdsa_signature output;
  CODE:
    if (!(secp256k1_ec_signature_serialize_compact(ctx,
                                                   &output,
                                                   &sig)))
      Perl_croak("Error serializing signature to compact");
    RETVAL = newSVpv((const char * const)&sig, sizeof(sig));
  OUTPUT:
    RETVAL

SV *
secp256k1_ecdsa_signature_serialize_der(ctx, sig)
    Crypt::secp256k1            ctx
    secp256k1_ecdsa_signature   sig
  PPCODE:
    char                      * output;
    size_t                      outputlen = 1024;
    Newx(output, outputlen, char); /* random number */
    if (!(secp256k1_ecdsa_signature_serialize_der(ctx, output, &outputlen, &sig)))
      Perl_croak("Error serializing signature to DER");
    sv_setpvn(ST(1), (char *)&output, outputlen);
    SvSETMAGIC(ST(1));
    Safefree(output);
    XSRETURN(1);

int
secp256k1_ecdsa_verify(ctx, sig, msg, pubkey)
    Crypt::secp256k1          ctx
    secp256k1_ecdsa_signature sig
    secp256k1_hash            msg
    secp256k1_pubkey          pubkey
  CODE:
    RETVAL = secp256k1_ecdsa_verify(ctx,
                                    (const secp256k1_ecdsa_signature *)&sig,
                                    (const secp256k1_blob32_cast)&msg,
                                    (const secp256k1_pubkey *)&pubkey);
  OUTPUT:
    RETVAL
